@extends('layouts.admin')

@section('title', 'Manage Images')

@section('styles')
<style>
    .folder-grid,
    .image-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
        gap: 1.5rem;
    }

    .folder-item,
    .image-item {
        min-width: 0;
        overflow: hidden;
    }

    .folder-item {
        background: #f8fafc;
        border-radius: 12px;
        padding: 1.5rem;
        text-align: center;
        transition: all 0.3s ease;
        position: relative;
        border: 1px solid #edf2f7;
    }

    .folder-item:hover {
        transform: translateY(-4px);
        background: #ffffff;
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
        border-color: var(--primary-red);
    }

    .image-item {
        background: #ffffff;
        border-radius: 12px;
        padding: 1rem;
        text-align: center;
        transition: all 0.3s ease;
        position: relative;
        border: 1px solid #edf2f7;
        box-shadow: 0 2px 5px rgba(0, 0, 0, 0.02);
    }

    .image-item:hover {
        transform: translateY(-4px);
        box-shadow: 0 12px 25px rgba(0, 0, 0, 0.08);
        border-color: #cbd5e0;
    }

    .folder-icon {
        font-size: 2.5rem;
        color: #f6ad55;
        margin-bottom: 0.75rem;
        display: block;
        transition: transform 0.3s;
    }

    .folder-name,
    .image-name {
        font-weight: 600;
        font-size: 0.95rem;
        color: #2d3748;
        display: block;
        word-break: break-all;
        margin-bottom: 0.5rem;
    }

    .image-thumb {
        width: 100%;
        height: 140px;
        object-fit: cover;
        border-radius: 8px;
        margin-bottom: 0.8rem;
        background: #f1f5f9;
    }

    .item-actions {
        display: flex;
        justify-content: center;
        gap: 8px;
        opacity: 0;
        transition: opacity 0.2s;
        position: absolute;
        top: 8px;
        right: 8px;
        background: rgba(255, 255, 255, 0.95);
        padding: 4px;
        border-radius: 6px;
        backdrop-filter: blur(4px);
        box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
        z-index: 10;
    }

    .folder-item:hover .item-actions,
    .image-item:hover .item-actions {
        opacity: 1;
    }

    .folder-count {
        font-size: 0.75rem;
        color: #718096;
        font-weight: 500;
    }
</style>
@endsection


@section('content')
@if(session('success'))
<div style="background: #d4edda; color: #155724; padding: 1rem; border-radius: 6px; margin-bottom: 1.5rem;">
    {{ session('success') }}
</div>
@endif

@if($errors->any())
<div style="background: #f8d7da; color: #721c24; padding: 1rem; border-radius: 6px; margin-bottom: 1.5rem;">
    <ul style="margin: 0;">
        @foreach($errors->all() as $error)
        <li>{{ $error }}</li>
        @endforeach
    </ul>
</div>
@endif

<div class="card">
    <div class="card-header">
        <h2 style="font-size: 1.2rem;">Folders</h2>
        <button class="btn btn-primary" onclick="openModal('folderModal')">Create Folder</button>
    </div>
    <div class="folder-grid">
        @forelse($folders as $folder)
        <div class="folder-item">
            <div class="item-actions">
                <button class="action-btn btn-edit" title="Edit Folder" onclick="editFolder('{{ $folder->id }}', '{{ $folder->name }}')">
                    <i class="fas fa-edit"></i>
                </button>
                <form action="{{ route('admin.images.folder.delete', $folder->id) }}" method="POST" onsubmit="return confirm('Delete folder and all images inside?')" style="display:inline;">
                    @csrf @method('DELETE')
                    <button type="submit" class="action-btn btn-delete" title="Delete Folder">
                        <i class="fas fa-trash"></i>
                    </button>
                </form>
            </div>
            <a href="{{ route('admin.images.show', $folder->slug) }}" style="text-decoration: none;">
                <i class="fas fa-folder folder-icon"></i>
                <span class="folder-name">{{ $folder->name }}</span>
            </a>
            <span style="font-size: 0.8rem; color: #718096;">({{ $folder->images->count() }} Images)</span>
        </div>
        @empty
        <p style="color: #666;">No folders created yet.</p>
        @endforelse
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h2 style="font-size: 1.2rem;">Root Images</h2>
        <button class="btn btn-primary" onclick="openModal('uploadModal')">
            <i class="fas fa-cloud-upload-alt"></i> Upload Images
        </button>
    </div>
    <div class="image-grid">
        @forelse($images as $image)
        <div class="image-item">
            <div class="item-actions">
                <button class="action-btn btn-move" title="Move Image" onclick="openMoveModal('{{ $image->id }}', '{{ $image->folder_id }}')">
                    <i class="fas fa-file-export"></i>
                </button>
                <form action="{{ route('admin.images.delete', $image->id) }}" method="POST" onsubmit="return confirm('Delete this image?')" style="display:inline;">
                    @csrf @method('DELETE')
                    <button type="submit" class="action-btn btn-delete" title="Delete Image">
                        <i class="fas fa-trash"></i>
                    </button>
                </form>
            </div>
            <img src="{{ asset($image->path) }}" class="image-thumb" alt="{{ $image->file_name }}">
            <span class="image-name" title="{{ $image->filename }}">{{ Str::limit($image->filename, 15) }}</span>
        </div>
        @empty
        <p style="color: #666;">No images uploaded in root folder.</p>
        @endforelse
    </div>
    <div class="pagination">
        {{ $images->links() }}
    </div>
</div>

<!-- Folder Modal -->
<div id="folderModal" class="modal">
    <div class="modal-content" style="max-width: 500px;">
        <div class="modal-header">
            <h3 id="folderModalTitle">Create Folder</h3>
            <button type="button" class="action-btn btn-secondary" onclick="closeModal('folderModal')" style="background:none; color:#64748b;">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form id="folderForm" action="{{ route('admin.images.folder.store') }}" method="POST">
            @csrf
            <div class="modal-body" style="padding: 2rem;">
                <div class="form-group">
                    <label class="form-label">Folder Name</label>
                    <input type="text" name="name" id="folderName" class="form-control" required placeholder="e.g. Summer Campaign 2024">
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('folderModal')">Cancel</button>
                <button type="submit" class="btn btn-primary">Save Changes</button>
            </div>
        </form>
    </div>
</div>

<!-- Upload Modal -->
<div id="uploadModal" class="modal">
    <div class="modal-content" style="max-width: 600px;">
        <div class="modal-header">
            <h3>Upload Images</h3>
            <button type="button" class="action-btn btn-secondary" onclick="closeModal('uploadModal')" style="background:none; color:#64748b;">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form action="{{ route('admin.images.upload') }}" method="POST" enctype="multipart/form-data">
            @csrf
            <div class="modal-body" style="padding: 2rem;">
                <div class="form-group">
                    <label class="form-label">Custom Prefix Name (Optional)</label>
                    <input type="text" name="custom_name" id="custom_name" class="form-control" placeholder="Leaving empty uses original filename">
                    <small style="color: #64748b;">Images will be renamed with this prefix (e.g. banner_1.jpg)</small>
                </div>
                <div class="form-group">
                    <label class="form-label">Select Files</label>
                    <div style="border: 2px dashed #e2e8f0; padding: 2rem; border-radius: 12px; text-align: center; background: #f8fafc;">
                        <input type="file" name="images[]" id="images" multiple required accept="image/*" style="opacity: 0; position: absolute; width: 0.1px; height: 0.1px;">
                        <label for="images" style="cursor: pointer;">
                            <i class="fas fa-cloud-upload-alt fa-3x" style="color: var(--primary-red); margin-bottom: 1rem;"></i>
                            <p style="font-weight: 600; color: #1e293b;">Click to select or drag and drop</p>
                            <p style="font-size: 0.8rem; color: #64748b;">PNG, JPG, SVG up to 5MB</p>
                        </label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('uploadModal')">Cancel</button>
                <button type="submit" class="btn btn-primary">Start Upload</button>
            </div>
        </form>
    </div>
</div>

<!-- Move Modal -->
<div id="moveModal" class="modal">
    <div class="modal-content" style="max-width: 500px;">
        <div class="modal-header">
            <h3>Move Image</h3>
            <button type="button" class="action-btn btn-secondary" onclick="closeModal('moveModal')" style="background:none; color:#64748b;">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form id="moveForm" action="" method="POST">
            @csrf @method('PUT')
            <div class="modal-body" style="padding: 2rem;">
                <div class="form-group">
                    <label class="form-label">Destination Folder</label>
                    <select name="folder_id" id="target_folder" class="form-control">
                        <option value="">Root Directory</option>
                        @foreach($folders as $f)
                        <option value="{{ $f->id }}">{{ $f->name }}</option>
                        @endforeach
                    </select>
                </div>
                <p style="color: #64748b; font-size: 0.85rem;">
                    <i class="fas fa-info-circle"></i> Moving an image will update its physical path on the server.
                </p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('moveModal')">Cancel</button>
                <button type="submit" class="btn btn-primary">Move Now</button>
            </div>
        </form>
    </div>
</div>

@endsection

@section('scripts')
<script>
    function openModal(id) {
        const modal = document.getElementById(id);
        modal.style.display = 'flex';
        setTimeout(() => modal.classList.add('show'), 10);
    }

    function closeModal(id) {
        const modal = document.getElementById(id);
        modal.classList.remove('show');
        setTimeout(() => modal.style.display = 'none', 300);
    }

    function editFolder(id, name) {
        document.getElementById('folderModalTitle').innerText = 'Edit Folder';
        document.getElementById('folderName').value = name;
        const form = document.getElementById('folderForm');
        form.action = "{{ url('admin/images/folder') }}/" + id;

        // Add hidden _method PUT
        let methodField = document.querySelector('input[name="_method"]');
        if (!methodField) {
            methodField = document.createElement('input');
            methodField.type = 'hidden';
            methodField.name = '_method';
            form.appendChild(methodField);
        }
        methodField.value = 'PUT';

        openModal('folderModal');
    }

    function openMoveModal(imageId, currentFolderId) {
        const form = document.getElementById('moveForm');
        form.action = "{{ url('admin/images') }}/" + imageId + "/move";
        document.getElementById('target_folder').value = currentFolderId || "";
        openModal('moveModal');
    }
</script>
@endsection