<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Build full URL for an image path. Uses APP_URL (backend) so images always load from the server that serves /uploads.
     */
    private function imageUrl(?string $path, Request $request): ?string
    {
        if (!$path || !is_string($path)) {
            return null;
        }
        $path = str_replace('\\', '/', trim($path, '/'));
        if ($path === '') {
            return null;
        }
        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));
        $base = rtrim(config('app.url'), '/');
        if ($base === '') {
            $base = rtrim($request->getSchemeAndHttpHost(), '/');
        }
        return $base . '/' . $encodedPath;
    }

    public function index(Request $request)
    {
        $query = Product::with(['thumbnail:id,path', 'images', 'category'])
            ->select('id', 'category_id', 'title', 'slug', 'thumbnail_id', 'price', 'discount_price', 'short_description', 'stock_quantity');

        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        $products = $query->paginate(12);

        // Explicitly set image URLs so they are always in the JSON response (dynamic URLs for product photos)
        $products->getCollection()->transform(function (Product $product) use ($request) {
            $thumbUrl = $this->imageUrl($product->thumbnail?->path, $request);
            $firstImage = $product->images->first();
            $firstUrl = $firstImage ? $this->imageUrl($firstImage->path, $request) : null;
            $product->setAttribute('thumbnail_url', $thumbUrl);
            $product->setAttribute('first_image_url', $firstUrl);
            $product->makeVisible(['thumbnail_url', 'first_image_url']);
            return $product;
        });

        return response()->json([
            'status' => 'success',
            'data' => $products
        ]);
    }

    public function show(Request $request, $slug)
    {
        $product = Product::where('slug', $slug)
            ->with(['thumbnail', 'images', 'tags', 'category', 'reviews' => function ($query) {
                $query->latest();
            }])
            ->firstOrFail();

        $product->setAttribute('thumbnail_url', $this->imageUrl($product->thumbnail?->path, $request));
        $firstImage = $product->images->first();
        $product->setAttribute('first_image_url', $firstImage ? $this->imageUrl($firstImage->path, $request) : null);
        $product->makeVisible(['thumbnail_url', 'first_image_url']);
        // Add full URL to each gallery image so frontend can display them (handles spaces in filenames)
        $product->images->transform(function ($img) use ($request) {
            $img->url = $this->imageUrl($img->path, $request);
            return $img;
        });

        return response()->json([
            'status' => 'success',
            'data' => $product
        ]);
    }
}
