<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ServiceArea;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ServiceAreaController extends Controller
{
    public function index()
    {
        $areas = ServiceArea::latest()->get();
        return view('admin.service-areas.index', compact('areas'));
    }

    public function create()
    {
        return view('admin.service-areas.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:service_areas,slug',
            'icon' => 'nullable|string',
            'short_description' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string',
            'meta_description' => 'nullable|string',
            'details' => 'nullable|array',
            'details.*.title' => 'required|string|max:255',
            'details.*.description' => 'nullable|string',
        ]);

        $data = $request->except('details');
        $data['slug'] = $data['slug'] ?: Str::slug($data['name']);

        $area = ServiceArea::create($data);

        if ($request->has('details')) {
            foreach ($request->details as $detail) {
                $area->details()->create($detail);
            }
        }

        return redirect()->route('admin.service-areas.index')->with('success', 'Service area created successfully.');
    }

    public function edit(ServiceArea $service_area)
    {
        $service_area->load('details');
        return view('admin.service-areas.edit', compact('service_area'));
    }

    public function update(Request $request, ServiceArea $service_area)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|unique:service_areas,slug,' . $service_area->id,
            'icon' => 'nullable|string',
            'short_description' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string',
            'meta_description' => 'nullable|string',
            'details' => 'nullable|array',
            'details.*.title' => 'required|string|max:255',
            'details.*.description' => 'nullable|string',
        ]);

        $data = $request->except('details');
        $data['slug'] = $data['slug'] ?: Str::slug($data['name']);

        $service_area->update($data);

        // Update details
        $service_area->details()->delete();
        if ($request->has('details')) {
            foreach ($request->details as $detail) {
                $service_area->details()->create($detail);
            }
        }

        return redirect()->route('admin.service-areas.index')->with('success', 'Service area updated successfully.');
    }

    public function destroy(ServiceArea $service_area)
    {
        $service_area->delete();
        return back()->with('success', 'Service area deleted successfully.');
    }
}
